<?php

declare(strict_types=1);

namespace Erlage\Photogram\Requests\Admin;

use Erlage\Photogram\System;
use Erlage\Photogram\Constants\ServerConstants;
use Erlage\Photogram\Constants\SystemConstants;
use Erlage\Photogram\Data\Models\Setting\SettingEnum;
use Erlage\Photogram\Data\Tables\Setting\SettingTable;

final class AdminSettingsValidator
{
    public static function validate(array $settingData): string
    {
        $value = $settingData[SettingTable::VALUE];

        switch ($settingData[SettingTable::KEY])
        {
            /*
            |--------------------------------------------------------------------------
            | content - uploads
            |--------------------------------------------------------------------------
            */

            case SystemConstants::SETTING_MAX_USER_DISPLAY_IMAGE_FILE_SIZE:
                if ( ! self::positiveIntValidator($value))
                {
                    return 'Maximum size allowed from User display image must be a positive integer.';
                }

            break;

            case SystemConstants::SETTING_SUPPORTED_USER_DISPLAY_IMAGE_FILE_FORMAT:
                if ( ! self::isValidCSV($value, array('jpg', 'webp', 'jpeg', 'png', 'bmp')))
                {
                    return 'Some of the extension in allowed User image are not supported.';
                }

            break;

            case SystemConstants::SETTING_MAX_POST_DISPLAY_CONTENT_ITEM_FILE_SIZE:
                if ( ! self::positiveIntValidator($value))
                {
                    return 'Maximum size allowed from Post display image must be a positive integer.';
                }

            break;

            case SystemConstants::SETTING_SUPPORTED_POST_DISPLAY_CONTENT_ITEM_FILE_FORMAT:
                if ( ! self::isValidCSV($value, array('jpg', 'webp', 'jpeg', 'png', 'bmp')))
                {
                    return 'Some of the extension in allowed Post image are not supported.';
                }

            break;

            case SystemConstants::SETTING_MAX_POST_DISPLAY_CONTENT_ITEM:
                if ( ! self::positiveIntValidator($value))
                {
                    return 'Maximum number of photos in a Post must be a positive integer.';
                }

            break;

            case SystemConstants::SETTING_MIN_POST_DISPLAY_CONTENT_ITEM:
                if ( ! self::positiveIntValidator($value))
                {
                    return 'Minimum number of photos in a Post must be a positive integer.';
                }

            break;

            /*
            |--------------------------------------------------------------------------
            | content - storage options
            |--------------------------------------------------------------------------
            */

            case ServerConstants::SS_ENUM_STORAGE_DISK_USER_IMAGES:
            case ServerConstants::SS_ENUM_STORAGE_DISK_POST_IMAGES:
            case ServerConstants::SS_ENUM_STORAGE_DISK_COLLECTION_IMAGES:
                if ( ! \in_array($value, array(
                    SettingEnum::STORAGE_DISK_LOCAL,
                    SettingEnum::STORAGE_DISK_AWS,
                    SettingEnum::STORAGE_DISK_AZURE_BLOB,
                    SettingEnum::STORAGE_DISK_WASABI,
                ))
                ) {
                    return 'Disk type is not implemented.';
                }

            break;

            case ServerConstants::SS_INT_COMPRESS_USER_DISPLAY_IMAGE_FILE_RES:
            case ServerConstants::SS_INT_COMPRESS_POST_DISPLAY_IMAGE_FILE_RES:
            case ServerConstants::SS_INT_COMPRESS_COLLECTION_COVER_IMAGE_FILE_RES:
                if ( ! self::positiveIntValidator($value))
                {
                    return 'Compression resolution must be a postive integer';
                }

                if ((int) $value > 1980)
                {
                    return 'Max compression resolution is 1980';
                }

            break;

            case ServerConstants::SS_INT_COMPRESS_USER_DISPLAY_IMAGE_FILE_QUALITY:
            case ServerConstants::SS_INT_COMPRESS_POST_DISPLAY_IMAGE_FILE_QUALITY:
            case ServerConstants::SS_INT_COMPRESS_COLLECTION_COVER_IMAGE_FILE_QUALITY:
                if ( ! self::positiveIntValidator($value))
                {
                    return 'Image quality value must be a postive integer';
                }

                if ((int) $value > 100 || (int) $value < 1)
                {
                    return 'Image quality value must be between 1 and 100';
                }

            break;

            /*
            |--------------------------------------------------------------------------
            | content - feeds tweaking
            |--------------------------------------------------------------------------
            */

            case ServerConstants::SS_BOOL_IMPLICIT_HASHTAG_FOLLOW_ON_POST_LIKE:
                if ( ! self::boolValidator($value))
                {
                    return 'Values you provided are not allowed. Please bug the auther if issue presists.';
                }

            break;

            case ServerConstants::SS_BOOL_IMPLICIT_HASHTAG_FOLLOW_ON_POST_COMMENT:
                if ( ! self::boolValidator($value))
                {
                    return 'Values you provided are not allowed. Please bug the auther if issue presists.';
                }

            break;

            case ServerConstants::SS_BOOL_IMPLICIT_HASHTAG_FOLLOW_ON_POST_COMMENT_REPLY:
                if ( ! self::boolValidator($value))
                {
                    return 'Values you provided are not allowed. Please bug the auther if issue presists.';
                }

            break;

            /*
            |--------------------------------------------------------------------------
            | security - email verification
            |--------------------------------------------------------------------------
            */

            case ServerConstants::SS_BOOL_USER_EMAIL_VERIFICATION:
                if ( ! self::boolValidator($value))
                {
                    return 'Email verification can either be on or off.';
                }

            break;

            /*
            |--------------------------------------------------------------------------
            | security - one time passwords
            |--------------------------------------------------------------------------
            */

            case ServerConstants::SS_INT_OTP_LIFETIME_IN_TRIES:
                if ( ! self::positiveIntValidator($value))
                {
                    return 'OTP Attempts allowed must be a positive number.';
                }

            break;

            case ServerConstants::SS_INT_OTP_LIFETIME_IN_HOURS:
                if ( ! self::positiveIntValidator($value))
                {
                    return 'OTP lifetime hours must be a positive number.';
                }

            break;

            /*
            |--------------------------------------------------------------------------
            | security - mailer settings
            |--------------------------------------------------------------------------
            */

            case ServerConstants::SS_TEXT_MAILER_CONTACT_EMAIL:
                if (false === \filter_var($value, FILTER_VALIDATE_EMAIL))
                {
                    return 'Contact email is not a valid email address.';
                }

            break;

            case ServerConstants::SS_TEXT_MAILER_NO_REPLY_EMAIL:
                if (false === \filter_var($value, FILTER_VALIDATE_EMAIL))
                {
                    return 'No-Reply email is not a valid email address.';
                }

            break;

            case ServerConstants::SS_BOOL_MAILER_SMTP:
                if ( ! self::boolValidator($value))
                {
                    return 'SMTP can either be on or off.';
                }

            break;

            case ServerConstants::SS_INT_MAILER_SMTP_PORT:
                $value = \trim($value);

                if (\strlen($value))
                {
                    if ( ! self::positiveIntValidator($value))
                    {
                        return 'Port must be a positive number.';
                    }
                }

            break;

            // case ServerConstants::SS_TEXT_MAILER_SMTP_HOST; break; // no-validation
            // case ServerConstants::SS_TEXT_MAILER_SMTP_USERNAME; break; // no-validation
            // case ServerConstants::SS_TEXT_MAILER_SMTP_PASSWORD; break; // no-validation

            /*
            |--------------------------------------------------------------------------
            | site settings
            |--------------------------------------------------------------------------
            */

            case ServerConstants::SS_TEXT_NAME:
                if ( ! \strlen($value))
                {
                    return 'Site name cannot be empty.';
                }

            break;

            case ServerConstants::SS_TEXT_API_IOS_UPDATE_URL:
            case ServerConstants::SS_TEXT_API_ANDROID_UPDATE_URL:
                if ( ! \strlen($value))
                {
                    return 'Update URL cannot be empty.';
                }

                if (false === \filter_var($value, FILTER_VALIDATE_URL) && 0 !== \strlen($value))
                {
                    return 'Update URL should be a valid web address.';
                }

                if ( ! System::str_starts_with($value, 'http') && ! System::str_starts_with($value, 'https'))
                {
                    return 'Update URL must starts with http/https, other schemes are not allowed on apps.';
                }

            break;

            case ServerConstants::SS_TEXT_API_UPDATE_MESSAGE:
                if ( ! \strlen($value))
                {
                    return 'Update message cannot be empty.';
                }

            break;
        }

        return SystemConstants::OK;
    }

    private static function boolValidator(string $value): bool
    {
        return (bool) (\in_array($value, array('1', '0')));
    }

    private static function positiveIntValidator(string $value): bool
    {
        $filter_options = array(
            'options' => array('min_range' => 1),
        );

        return false !== \filter_var($value, FILTER_VALIDATE_INT, $filter_options);
    }

    private static function isValidCSV(string $value, array $allowed): bool
    {
        $explode = \array_diff(\array_map('trim', \explode(',', $value)), array(''));

        return 0 == \count(\array_diff($explode, $allowed));
    }
}
